#!/usr/bin/env bash

cd "$(dirname "$(readlink -f "${0}")")" || exit 1

ask_continue() {
    read -n1 -rp  "$1 ([y]es or [N]o): " REPLY
    echo
    case ${REPLY,} in
        y) ;;
        *)     echo "Installation canceled." && exit 1;;
    esac
}

PRODUCT=Charlatan3

! ./check_compatibility.sh && ask_continue "Do you want to continue with installation anyway?"

more license.txt
ask_continue "Do you accept the license terms?"

clap_dir=${CLAP_PATH/:*}
clap_dir=${clap_dir:-$HOME/.clap}
vst3_dir="$HOME/.vst3"
DATA_DIR="${XDG_DATA_HOME:-$HOME/.local/share}"
DATA_DIR="$DATA_DIR/charlatan"

clear
echo "CLAP plug-in will be installed to $clap_dir"
echo "VST3 plug-in will be installed to $vst3_dir"
echo "Factory presets will be installed to $DATA_DIR"
echo "WARNING: Existing Factory presets will be overwritten!"
echo "         To prevent data loss, backup your presets directory before proceeding."
ask_continue "Do you want to continue with installation?"

mkdir -p "$clap_dir"
cp $PRODUCT.so "$clap_dir/$PRODUCT.clap"
mkdir -p "$vst3_dir/$PRODUCT.vst3/Contents/x86_64-linux"
cp $PRODUCT.so "$vst3_dir/$PRODUCT.vst3/Contents/x86_64-linux/"
mkdir -p "$DATA_DIR"
cp -rf presets "$DATA_DIR/"

if command -v flatpak >/dev/null 2>&1; then
	known_apps=(
		com.bitwig.BitwigStudio
		fm.reaper.Reaper
		org.ardour.Ardour
		org.rncbc.qtractor
	)

	found_apps=$(comm -12 <(printf '%s\n' "${known_apps[@]}") <(flatpak list --app --columns=application | sort))

	if [ -n "$found_apps" ]; then
		clear
		echo "The following Flatpak applications installed on your system are known to support audio plugins."
		for app in $found_apps; do
			echo "- $app"
		done
		echo
		echo "The following adjustments need to be performed to allow Charlatan accessing your presets folder and work properly overall:"
		echo "- Add user permission to access desktop session bus (D-Bus)"
		echo "- Create symlink from sandboxed environment to your user preset directory."
		echo
		read -n1 -rp "Do you want me to make the adjustments? ([Y]es or [n]o): " REPLY
		echo
		case ${REPLY,} in
			n) echo "Skipped"; exit 0;;
			*)
		esac

		for app in $found_apps; do
			./flatpak_fix.sh "$app"
		done
	fi
fi
